'use client';

import { useState, useRef } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { X, Upload, Link as LinkIcon } from 'lucide-react';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';

export default function ImageUploader({ images = [], onChange, maxImages = 5, label = 'تصاویر' }) {
  const [imageUrls, setImageUrls] = useState(images);
  const [urlInput, setUrlInput] = useState('');
  const fileInputRef = useRef(null);

  const handleFileUpload = async (e) => {
    const files = Array.from(e.target.files || []);
    
    for (const file of files) {
      if (imageUrls.length >= maxImages) {
        alert(`حداکثر ${maxImages} تصویر مجاز است`);
        break;
      }

      if (file.size > 2 * 1024 * 1024) { // 2MB limit
        alert('حجم فایل نباید بیشتر از 2 مگابایت باشد');
        continue;
      }

      // Convert to base64
      const reader = new FileReader();
      reader.onload = (event) => {
        const base64 = event.target.result;
        const newImages = [...imageUrls, base64];
        setImageUrls(newImages);
        onChange(newImages);
      };
      reader.readAsDataURL(file);
    }
    
    // Reset input
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  const handleUrlAdd = () => {
    if (!urlInput.trim()) return;
    
    if (imageUrls.length >= maxImages) {
      alert(`حداکثر ${maxImages} تصویر مجاز است`);
      return;
    }

    const newImages = [...imageUrls, urlInput.trim()];
    setImageUrls(newImages);
    onChange(newImages);
    setUrlInput('');
  };

  const handleRemove = (index) => {
    const newImages = imageUrls.filter((_, i) => i !== index);
    setImageUrls(newImages);
    onChange(newImages);
  };

  return (
    <div className="space-y-4">
      <Label>{label}</Label>
      
      <Tabs defaultValue="upload" className="w-full">
        <TabsList className="grid w-full grid-cols-2">
          <TabsTrigger value="upload">آپلود فایل</TabsTrigger>
          <TabsTrigger value="url">لینک تصویر</TabsTrigger>
        </TabsList>
        
        <TabsContent value="upload" className="space-y-2">
          <div className="flex items-center gap-2">
            <Input
              ref={fileInputRef}
              type="file"
              accept="image/*"
              onChange={handleFileUpload}
              multiple
              className="hidden"
              id="file-upload"
            />
            <Button
              type="button"
              variant="outline"
              onClick={() => fileInputRef.current?.click()}
              disabled={imageUrls.length >= maxImages}
              className="w-full"
            >
              <Upload className="h-4 w-4 mr-2" />
              انتخاب تصویر از کامپیوتر
            </Button>
          </div>
          <p className="text-xs text-muted-foreground">
            فرمت‌های مجاز: JPG, PNG, WebP | حداکثر حجم: 2MB
          </p>
        </TabsContent>
        
        <TabsContent value="url" className="space-y-2">
          <div className="flex gap-2">
            <Input
              placeholder="https://example.com/image.jpg"
              value={urlInput}
              onChange={(e) => setUrlInput(e.target.value)}
              onKeyPress={(e) => e.key === 'Enter' && handleUrlAdd()}
            />
            <Button
              type="button"
              onClick={handleUrlAdd}
              disabled={imageUrls.length >= maxImages}
            >
              <LinkIcon className="h-4 w-4" />
            </Button>
          </div>
        </TabsContent>
      </Tabs>

      {/* Image Preview */}
      {imageUrls.length > 0 && (
        <div className="grid grid-cols-3 gap-4">
          {imageUrls.map((img, index) => (
            <div key={index} className="relative group">
              <div className="aspect-square bg-muted rounded-lg overflow-hidden border-2 border-border">
                <img
                  src={img}
                  alt={`تصویر ${index + 1}`}
                  className="w-full h-full object-cover"
                  onError={(e) => {
                    e.target.src = 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="100" height="100"%3E%3Crect fill="%23ddd" width="100" height="100"/%3E%3Ctext fill="%23999" x="50%25" y="50%25" text-anchor="middle" dy=".3em"%3ENo Image%3C/text%3E%3C/svg%3E';
                  }}
                />
              </div>
              <Button
                type="button"
                size="icon"
                variant="destructive"
                className="absolute top-2 right-2 h-6 w-6 opacity-0 group-hover:opacity-100 transition-opacity"
                onClick={() => handleRemove(index)}
              >
                <X className="h-3 w-3" />
              </Button>
              <div className="absolute bottom-2 left-2 bg-black/70 text-white text-xs px-2 py-1 rounded">
                {index + 1}
              </div>
            </div>
          ))}
        </div>
      )}
      
      <p className="text-xs text-muted-foreground">
        {imageUrls.length} از {maxImages} تصویر
      </p>
    </div>
  );
}