'use client';

import { useState } from 'react';
import { useTranslations } from 'next-intl';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Carousel, CarouselContent, CarouselItem, CarouselNext, CarouselPrevious } from '@/components/ui/carousel';
import { Plus, Minus, X } from 'lucide-react';
import { Alert, AlertDescription } from '@/components/ui/alert';

export default function ProductModal({ product, onClose, onOrderComplete, locale, userWallet }) {
  const t = useTranslations();
  const [quantities, setQuantities] = useState({});
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [success, setSuccess] = useState(false);

  const name = locale === 'fa' ? product.nameFA : product.nameEN;
  const images = product.images || [product.image];

  const updateQuantity = (variantId, delta) => {
    setQuantities(prev => ({
      ...prev,
      [variantId]: Math.max(0, (prev[variantId] || 0) + delta)
    }));
  };

  const calculateTotal = () => {
    let totalGrams = 0;
    product.variants?.forEach(variant => {
      const qty = quantities[variant.id] || 0;
      if (qty > 0) {
        totalGrams += variant.weight * qty;
      }
    });
    return totalGrams;
  };

  const handleOrder = async () => {
    const totalGrams = calculateTotal();
    
    if (totalGrams === 0) {
      setError('لطفاً حداقل یک محصول انتخاب کنید');
      return;
    }

    if (totalGrams > userWallet) {
      setError(`موجودی کیف پول شما کافی نیست. موجودی فعلی: ${userWallet.toFixed(2)} گرم`);
      return;
    }

    setLoading(true);
    setError('');

    try {
      const token = localStorage.getItem('token');
      const orderItems = product.variants
        .filter(variant => (quantities[variant.id] || 0) > 0)
        .map(variant => ({
          productId: product.id,
          size: variant.size,
          quantity: quantities[variant.id],
          karatType: variant.karats,
          gramsPerUnit: variant.weight,
          totalGrams: variant.weight * quantities[variant.id]
        }));

      const response = await fetch('/api/orders', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          items: orderItems,
          totalGoldGrams: totalGrams
        })
      });

      const data = await response.json();

      if (response.ok) {
        setSuccess(true);
        setTimeout(() => {
          onOrderComplete();
          onClose();
        }, 1500);
      } else {
        setError(data.error || 'خطا در ثبت سفارش');
      }
    } catch (error) {
      setError('خطای شبکه. لطفاً دوباره تلاش کنید.');
    } finally {
      setLoading(false);
    }
  };

  const totalGrams = calculateTotal();

  return (
    <Dialog open={true} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center justify-between">
            <span>{name}</span>
            {product.code && (
              <Badge variant="outline">{product.code}</Badge>
            )}
          </DialogTitle>
        </DialogHeader>

        {success && (
          <Alert>
            <AlertDescription>سفارش شما با موفقیت ثبت شد!</AlertDescription>
          </Alert>
        )}

        {error && (
          <Alert variant="destructive">
            <AlertDescription>{error}</AlertDescription>
          </Alert>
        )}

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          {/* Images Carousel */}
          <div className="relative">
            {images && images.length > 0 ? (
              <Carousel className="w-full">
                <CarouselContent>
                  {images.map((image, index) => (
                    <CarouselItem key={index}>
                      <div className="aspect-square bg-muted rounded-lg overflow-hidden">
                        <img 
                          src={image} 
                          alt={`${name} ${index + 1}`}
                          className="w-full h-full object-cover"
                        />
                      </div>
                    </CarouselItem>
                  ))}
                </CarouselContent>
                {images.length > 1 && (
                  <>
                    <CarouselPrevious />
                    <CarouselNext />
                  </>
                )}
              </Carousel>
            ) : (
              <div className="aspect-square bg-muted rounded-lg flex items-center justify-center">
                <span className="text-muted-foreground">No Image</span>
              </div>
            )}
          </div>

          {/* Product Info & Variants */}
          <div className="space-y-6">
            <div>
              <h3 className="text-2xl font-bold mb-2">{name}</h3>
              {product.code && (
                <p className="text-muted-foreground">{t('products.code')}: {product.code}</p>
              )}
            </div>

            {/* Variants */}
            {product.variants && product.variants.length > 0 && (
              <div className="space-y-4">
                <h4 className="font-semibold">انتخاب سایز و تعداد:</h4>
                {product.variants.map((variant) => (
                  <div key={variant.id} className="flex items-center justify-between p-3 border rounded-lg">
                    <div className="flex-1">
                      <div className="font-medium">
                        {variant.size} ({variant.weight} {t('common.gram')})
                      </div>
                      <div className="text-sm text-muted-foreground">
                        {variant.karats}
                      </div>
                    </div>
                    
                    <div className="flex items-center gap-2">
                      <Button
                        size="icon"
                        variant="outline"
                        onClick={() => updateQuantity(variant.id, -1)}
                        disabled={loading}
                      >
                        <Minus className="h-4 w-4" />
                      </Button>
                      <span className="w-12 text-center font-medium">
                        {quantities[variant.id] || 0}
                      </span>
                      <Button
                        size="icon"
                        variant="outline"
                        onClick={() => updateQuantity(variant.id, 1)}
                        disabled={loading}
                      >
                        <Plus className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                ))}
              </div>
            )}

            {/* Total */}
            <div className="p-4 bg-muted rounded-lg">
              <div className="flex justify-between items-center mb-2">
                <span className="text-muted-foreground">مجموع وزن:</span>
                <span className="text-xl font-bold">{totalGrams.toFixed(2)} {t('common.gram')}</span>
              </div>
              <div className="flex justify-between items-center text-sm">
                <span className="text-muted-foreground">موجودی کیف پول:</span>
                <span className={totalGrams > userWallet ? 'text-destructive' : 'text-muted-foreground'}>
                  {userWallet.toFixed(2)} {t('common.gram')}
                </span>
              </div>
            </div>
          </div>
        </div>

        <DialogFooter>
          <Button variant="outline" onClick={onClose} disabled={loading}>
            {t('common.cancel')}
          </Button>
          <Button onClick={handleOrder} disabled={loading || totalGrams === 0}>
            {loading ? t('common.loading') : t('products.orderNow')}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
