'use client';

import { useState } from 'react';
import { useTranslations } from 'next-intl';
import { useRouter, usePathname } from 'next/navigation';
import { ChevronRight, ChevronLeft, LayoutDashboard, ShoppingCart, FileText, Settings, Printer } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { cn } from '@/lib/utils';

export default function Sidebar({ user, onLogout, locale }) {
  const t = useTranslations();
  const router = useRouter();
  const pathname = usePathname();
  const [collapsed, setCollapsed] = useState(false);

  const menuItems = [
    { icon: LayoutDashboard, label: t('common.dashboard'), path: `/${locale}`, section: null },
    { icon: ShoppingCart, label: t('common.orders'), path: `/${locale}/orders`, section: 'nokia' },
    { icon: Printer, label: t('common.printLogs'), path: `/${locale}/print-logs`, section: 'control' },
    ...(user?.role === 'admin' ? [{ icon: Settings, label: t('common.admin'), path: `/${locale}/admin`, section: 'control' }] : []),
  ];

  const isRTL = locale === 'fa';
  const CollapseIcon = isRTL ? ChevronRight : ChevronLeft;

  return (
    <div 
      className={cn(
        "bg-sidebar border-border flex flex-col transition-all duration-300",
        collapsed ? "w-16" : "w-64",
        isRTL ? "border-l" : "border-r"
      )}
    >
      {/* Logo & Collapse */}
      <div className="p-4 flex items-center justify-between border-b border-border">
        {!collapsed && (
          <div className="flex items-center gap-2">
            <div className="w-8 h-8 bg-primary rounded-lg flex items-center justify-center">
              <span className="text-primary-foreground font-bold text-lg">N</span>
            </div>
            <span className="font-bold text-xl">NovaFab</span>
          </div>
        )}
        <Button 
          variant="ghost" 
          size="icon"
          onClick={() => setCollapsed(!collapsed)}
          className={cn(collapsed && "mx-auto")}
        >
          <CollapseIcon className="h-5 w-5" />
        </Button>
      </div>

      {/* Credit Display */}
      {!collapsed && user && (
        <div className="p-4 border-b border-border">
          <div className="text-sm text-muted-foreground">{t('common.credit')}</div>
          <div className="text-2xl font-bold">
            {user.walletGrams?.toFixed(2) || '0.00'}
          </div>
          <div className="text-xs text-muted-foreground">{t('common.gram')}</div>
        </div>
      )}

      {/* Menu Items */}
      <nav className="flex-1 p-2">
        {/* NOKIA MARKET Section */}
        {!collapsed && (
          <div className="px-3 py-2 text-xs font-semibold text-muted-foreground uppercase">
            NOKIA MARKET
          </div>
        )}
        
        {menuItems.filter(item => item.section === 'nokia' || item.section === null).map((item) => {
          const Icon = item.icon;
          const isActive = pathname === item.path;
          
          return (
            <Button
              key={item.path}
              variant={isActive ? "secondary" : "ghost"}
              className={cn(
                "w-full justify-start mb-1",
                collapsed && "justify-center px-2"
              )}
              onClick={() => router.push(item.path)}
            >
              <Icon className={cn("h-5 w-5", !collapsed && (isRTL ? "ml-2" : "mr-2"))} />
              {!collapsed && <span>{item.label}</span>}
            </Button>
          );
        })}

        {/* CONTROL PANEL Section */}
        {!collapsed && (
          <div className="px-3 py-2 mt-4 text-xs font-semibold text-muted-foreground uppercase">
            CONTROL PANEL
          </div>
        )}
        
        {menuItems.filter(item => item.section === 'control').map((item) => {
          const Icon = item.icon;
          const isActive = pathname === item.path;
          
          return (
            <Button
              key={item.path}
              variant={isActive ? "secondary" : "ghost"}
              className={cn(
                "w-full justify-start mb-1",
                collapsed && "justify-center px-2"
              )}
              onClick={() => router.push(item.path)}
            >
              <Icon className={cn("h-5 w-5", !collapsed && (isRTL ? "ml-2" : "mr-2"))} />
              {!collapsed && <span>{item.label}</span>}
            </Button>
          );
        })}
      </nav>
    </div>
  );
}