'use client';

import { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card } from '@/components/ui/card';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/dialog';
import { Trash2, Edit, Plus } from 'lucide-react';
import { Textarea } from '@/components/ui/textarea';

export default function CategoriesManager({ locale }) {
  const [categories, setCategories] = useState([]);
  const [subcategories, setSubcategories] = useState([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [modalType, setModalType] = useState('category'); // 'category' or 'subcategory'
  const [selectedCategory, setSelectedCategory] = useState(null);
  const [formData, setFormData] = useState({
    nameFA: '',
    nameEN: '',
    image: '',
    averageWeight: 0,
    order: 0
  });

  useEffect(() => {
    loadCategories();
    loadSubcategories();
  }, []);

  const loadCategories = async () => {
    try {
      const response = await fetch('/api/categories');
      const data = await response.json();
      setCategories(data.categories || []);
    } catch (error) {
      console.error('Error loading categories:', error);
    } finally {
      setLoading(false);
    }
  };

  const loadSubcategories = async () => {
    try {
      const response = await fetch('/api/subcategories');
      const data = await response.json();
      setSubcategories(data.subcategories || []);
    } catch (error) {
      console.error('Error loading subcategories:', error);
    }
  };

  const handleSave = async () => {
    try {
      const token = localStorage.getItem('token');
      const endpoint = modalType === 'category' ? '/api/categories' : '/api/subcategories';
      
      const body = modalType === 'category' ? {
        nameFA: formData.nameFA,
        nameEN: formData.nameEN,
        image: formData.image || null,
        order: parseInt(formData.order) || 0
      } : {
        categoryId: selectedCategory,
        nameFA: formData.nameFA,
        nameEN: formData.nameEN,
        image: formData.image || null,
        averageWeight: parseFloat(formData.averageWeight) || 0,
        order: parseInt(formData.order) || 0
      };

      await fetch(endpoint, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(body)
      });

      setShowModal(false);
      resetForm();
      loadCategories();
      loadSubcategories();
    } catch (error) {
      console.error('Error saving:', error);
    }
  };

  const handleDelete = async (id, type) => {
    if (!confirm('آیا مطمئن هستید؟')) return;

    try {
      const token = localStorage.getItem('token');
      const endpoint = type === 'category' ? `/api/categories/${id}` : `/api/subcategories/${id}`;
      
      await fetch(endpoint, {
        method: 'DELETE',
        headers: { 'Authorization': `Bearer ${token}` }
      });

      loadCategories();
      loadSubcategories();
    } catch (error) {
      console.error('Error deleting:', error);
    }
  };

  const resetForm = () => {
    setFormData({
      nameFA: '',
      nameEN: '',
      image: '',
      averageWeight: 0,
      order: 0
    });
    setSelectedCategory(null);
  };

  if (loading) {
    return <div className="text-center py-8">در حال بارگذاری...</div>;
  }

  return (
    <div className="space-y-6">
      {/* Categories */}
      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <h2 className="text-2xl font-bold">دسته‌بندی‌ها</h2>
          <Button onClick={() => {
            setModalType('category');
            setShowModal(true);
          }}>
            <Plus className="h-4 w-4 mr-2" />
            افزودن دسته‌بندی
          </Button>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          {categories.map((cat) => (
            <div key={cat.id} className="border rounded-lg p-4">
              <div className="flex items-start justify-between">
                <div className="flex-1">
                  <h3 className="font-bold text-lg">{cat.nameFA}</h3>
                  <p className="text-sm text-muted-foreground">{cat.nameEN}</p>
                </div>
                <Button
                  size="sm"
                  variant="destructive"
                  onClick={() => handleDelete(cat.id, 'category')}
                >
                  <Trash2 className="h-4 w-4" />
                </Button>
              </div>
              
              <div className="mt-4">
                <Button
                  size="sm"
                  variant="outline"
                  className="w-full"
                  onClick={() => {
                    setModalType('subcategory');
                    setSelectedCategory(cat.id);
                    setShowModal(true);
                  }}
                >
                  <Plus className="h-4 w-4 mr-2" />
                  افزودن زیردسته
                </Button>
              </div>

              {/* Subcategories */}
              <div className="mt-4 space-y-2">
                {subcategories
                  .filter(sub => sub.categoryId === cat.id)
                  .map(sub => (
                    <div key={sub.id} className="flex items-center justify-between bg-muted p-2 rounded">
                      <div className="text-sm">
                        <div className="font-medium">{sub.nameFA}</div>
                        <div className="text-xs text-muted-foreground">
                          میانگین: {sub.averageWeight} گرم
                        </div>
                      </div>
                      <Button
                        size="sm"
                        variant="ghost"
                        onClick={() => handleDelete(sub.id, 'subcategory')}
                      >
                        <Trash2 className="h-3 w-3" />
                      </Button>
                    </div>
                  ))}
              </div>
            </div>
          ))}
        </div>
      </Card>

      {/* Add/Edit Modal */}
      <Dialog open={showModal} onOpenChange={setShowModal}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>
              {modalType === 'category' ? 'افزودن دسته‌بندی' : 'افزودن زیردسته'}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div>
              <Label>نام فارسی</Label>
              <Input
                value={formData.nameFA}
                onChange={(e) => setFormData({ ...formData, nameFA: e.target.value })}
                placeholder="مثال: انگشتر"
              />
            </div>

            <div>
              <Label>نام انگلیسی</Label>
              <Input
                value={formData.nameEN}
                onChange={(e) => setFormData({ ...formData, nameEN: e.target.value })}
                placeholder="Example: Ring"
              />
            </div>

            {modalType === 'subcategory' && (
              <div>
                <Label>میانگین وزن (گرم)</Label>
                <Input
                  type="number"
                  value={formData.averageWeight}
                  onChange={(e) => setFormData({ ...formData, averageWeight: e.target.value })}
                  placeholder="12.5"
                  step="0.1"
                />
              </div>
            )}

            <div>
              <Label>ترتیب نمایش</Label>
              <Input
                type="number"
                value={formData.order}
                onChange={(e) => setFormData({ ...formData, order: e.target.value })}
                placeholder="1"
              />
            </div>

            <div>
              <Label>آدرس تصویر (اختیاری)</Label>
              <Textarea
                value={formData.image}
                onChange={(e) => setFormData({ ...formData, image: e.target.value })}
                placeholder="https://... یا base64"
                rows={3}
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setShowModal(false);
              resetForm();
            }}>
              انصراف
            </Button>
            <Button onClick={handleSave}>
              ذخیره
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}