'use client';

import { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card } from '@/components/ui/card';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Checkbox } from '@/components/ui/checkbox';
import { Trash2, Plus } from 'lucide-react';

export default function ProductsManager({ locale }) {
  const [products, setProducts] = useState([]);
  const [categories, setCategories] = useState([]);
  const [subcategories, setSubcategories] = useState([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [formData, setFormData] = useState({
    subcategoryId: '',
    code: '',
    nameFA: '',
    nameEN: '',
    images: '',
    weights: {
      '9k': '',
      '14k': '',
      '18k': '',
      '21k': '',
      '22k': ''
    },
    hasStone: false,
    variants: []
  });
  const [newVariant, setNewVariant] = useState({ size: '', weight: '', karats: '14K' });

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    try {
      const [productsRes, categoriesRes, subcategoriesRes] = await Promise.all([
        fetch('/api/products'),
        fetch('/api/categories'),
        fetch('/api/subcategories')
      ]);

      const productsData = await productsRes.json();
      const categoriesData = await categoriesRes.json();
      const subcategoriesData = await subcategoriesRes.json();

      setProducts(productsData.products || []);
      setCategories(categoriesData.categories || []);
      setSubcategories(subcategoriesData.subcategories || []);
    } catch (error) {
      console.error('Error loading data:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleSave = async () => {
    try {
      const token = localStorage.getItem('token');
      
      // Parse images (comma-separated URLs or base64)
      const images = formData.images.split(',').map(img => img.trim()).filter(img => img);

      // Convert weights to numbers
      const weights = {};
      Object.keys(formData.weights).forEach(karat => {
        const value = parseFloat(formData.weights[karat]);
        if (value > 0) weights[karat] = value;
      });

      await fetch('/api/products', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          subcategoryId: formData.subcategoryId,
          code: formData.code,
          nameFA: formData.nameFA,
          nameEN: formData.nameEN,
          images,
          weights,
          hasStone: formData.hasStone,
          variants: formData.variants
        })
      });

      setShowModal(false);
      resetForm();
      loadData();
    } catch (error) {
      console.error('Error saving product:', error);
    }
  };

  const handleDelete = async (id) => {
    if (!confirm('آیا مطمئن هستید؟')) return;

    try {
      const token = localStorage.getItem('token');
      await fetch(`/api/products/${id}`, {
        method: 'DELETE',
        headers: { 'Authorization': `Bearer ${token}` }
      });
      loadData();
    } catch (error) {
      console.error('Error deleting product:', error);
    }
  };

  const addVariant = () => {
    if (newVariant.size && newVariant.weight) {
      setFormData({
        ...formData,
        variants: [...formData.variants, { ...newVariant, weight: parseFloat(newVariant.weight) }]
      });
      setNewVariant({ size: '', weight: '', karats: '14K' });
    }
  };

  const removeVariant = (index) => {
    setFormData({
      ...formData,
      variants: formData.variants.filter((_, i) => i !== index)
    });
  };

  const resetForm = () => {
    setFormData({
      subcategoryId: '',
      code: '',
      nameFA: '',
      nameEN: '',
      images: '',
      weights: { '9k': '', '14k': '', '18k': '', '21k': '', '22k': '' },
      hasStone: false,
      variants: []
    });
  };

  if (loading) {
    return <div className="text-center py-8">در حال بارگذاری...</div>;
  }

  return (
    <Card className="p-6">
      <div className="flex items-center justify-between mb-4">
        <h2 className="text-2xl font-bold">مدیریت محصولات</h2>
        <Button onClick={() => setShowModal(true)}>
          <Plus className="h-4 w-4 mr-2" />
          افزودن محصول
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-4 gap-4">
        {products.map((product) => {
          const subcategory = subcategories.find(s => s.id === product.subcategoryId);
          return (
            <div key={product.id} className="border rounded-lg p-4">
              {product.images && product.images[0] && (
                <div className="aspect-square bg-muted rounded mb-3 overflow-hidden">
                  <img src={product.images[0]} alt={product.nameFA} className="w-full h-full object-cover" />
                </div>
              )}
              <div className="space-y-2">
                <div>
                  <div className="font-bold">{product.nameFA}</div>
                  <div className="text-sm text-muted-foreground">{product.code}</div>
                </div>
                <div className="text-xs text-muted-foreground">
                  {subcategory?.nameFA}
                </div>
                <div className="flex items-center gap-2">
                  {product.hasStone && (
                    <span className="text-xs bg-primary/10 text-primary px-2 py-1 rounded">سنگ‌دار</span>
                  )}
                  {product.variants && product.variants.length > 0 && (
                    <span className="text-xs bg-secondary px-2 py-1 rounded">
                      {product.variants.length} سایز
                    </span>
                  )}
                </div>
                <Button
                  size="sm"
                  variant="destructive"
                  className="w-full"
                  onClick={() => handleDelete(product.id)}
                >
                  <Trash2 className="h-3 w-3 mr-2" />
                  حذف
                </Button>
              </div>
            </div>
          );
        })}
      </div>

      {/* Add Product Modal */}
      <Dialog open={showModal} onOpenChange={setShowModal}>
        <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>افزودن محصول جدید</DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label>زیردسته</Label>
                <Select value={formData.subcategoryId} onValueChange={(value) => setFormData({ ...formData, subcategoryId: value })}>
                  <SelectTrigger>
                    <SelectValue placeholder="انتخاب زیردسته" />
                  </SelectTrigger>
                  <SelectContent>
                    {subcategories.map(sub => (
                      <SelectItem key={sub.id} value={sub.id}>{sub.nameFA}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label>کد محصول</Label>
                <Input
                  value={formData.code}
                  onChange={(e) => setFormData({ ...formData, code: e.target.value })}
                  placeholder="BLV05"
                />
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label>نام فارسی</Label>
                <Input
                  value={formData.nameFA}
                  onChange={(e) => setFormData({ ...formData, nameFA: e.target.value })}
                  placeholder="دستبند طلا"
                />
              </div>

              <div>
                <Label>نام انگلیسی</Label>
                <Input
                  value={formData.nameEN}
                  onChange={(e) => setFormData({ ...formData, nameEN: e.target.value })}
                  placeholder="Gold Bracelet"
                />
              </div>
            </div>

            <div>
              <Label>تصاویر (جدا شده با کاما)</Label>
              <Textarea
                value={formData.images}
                onChange={(e) => setFormData({ ...formData, images: e.target.value })}
                placeholder="https://example.com/image1.jpg, https://example.com/image2.jpg"
                rows={2}
              />
            </div>

            <div>
              <Label>وزن بر اساس عیار (گرم)</Label>
              <div className="grid grid-cols-5 gap-2">
                {['9k', '14k', '18k', '21k', '22k'].map(karat => (
                  <div key={karat}>
                    <Label className="text-xs">{karat.toUpperCase()}</Label>
                    <Input
                      type="number"
                      value={formData.weights[karat]}
                      onChange={(e) => setFormData({
                        ...formData,
                        weights: { ...formData.weights, [karat]: e.target.value }
                      })}
                      placeholder="10.5"
                      step="0.1"
                    />
                  </div>
                ))}
              </div>
            </div>

            <div className="flex items-center space-x-2 space-x-reverse">
              <Checkbox
                id="hasStone"
                checked={formData.hasStone}
                onCheckedChange={(checked) => setFormData({ ...formData, hasStone: checked })}
              />
              <Label htmlFor="hasStone">سنگ‌دار</Label>
            </div>

            {/* Variants */}
            <div>
              <Label>سایزها و وزن‌ها</Label>
              <div className="space-y-2 mt-2">
                {formData.variants.map((variant, index) => (
                  <div key={index} className="flex items-center gap-2 bg-muted p-2 rounded">
                    <span className="text-sm flex-1">{variant.size} - {variant.weight}g ({variant.karats})</span>
                    <Button size="sm" variant="ghost" onClick={() => removeVariant(index)}>
                      <Trash2 className="h-3 w-3" />
                    </Button>
                  </div>
                ))}
              </div>
              
              <div className="grid grid-cols-4 gap-2 mt-3">
                <Input
                  placeholder="سایز (مثلا B)"
                  value={newVariant.size}
                  onChange={(e) => setNewVariant({ ...newVariant, size: e.target.value })}
                />
                <Input
                  type="number"
                  placeholder="وزن"
                  value={newVariant.weight}
                  onChange={(e) => setNewVariant({ ...newVariant, weight: e.target.value })}
                  step="0.01"
                />
                <Select value={newVariant.karats} onValueChange={(value) => setNewVariant({ ...newVariant, karats: value })}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="9K">9K</SelectItem>
                    <SelectItem value="14K">14K</SelectItem>
                    <SelectItem value="18K">18K</SelectItem>
                    <SelectItem value="21K">21K</SelectItem>
                    <SelectItem value="22K">22K</SelectItem>
                  </SelectContent>
                </Select>
                <Button size="sm" onClick={addVariant}>
                  <Plus className="h-4 w-4" />
                </Button>
              </div>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              setShowModal(false);
              resetForm();
            }}>
              انصراف
            </Button>
            <Button onClick={handleSave}>
              ذخیره
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </Card>
  );
}