'use client';

import { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { CheckCircle, XCircle, Wallet } from 'lucide-react';

export default function UsersManager({ locale }) {
  const [users, setUsers] = useState([]);
  const [loading, setLoading] = useState(true);
  const [selectedUser, setSelectedUser] = useState(null);
  const [showWalletModal, setShowWalletModal] = useState(false);
  const [walletAmount, setWalletAmount] = useState('');
  const [walletOperation, setWalletOperation] = useState('add');

  useEffect(() => {
    loadUsers();
  }, []);

  const loadUsers = async () => {
    try {
      const token = localStorage.getItem('token');
      const response = await fetch('/api/users', {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      const data = await response.json();
      setUsers(data.users || []);
    } catch (error) {
      console.error('Error loading users:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleApproveUser = async (userId, status) => {
    try {
      const token = localStorage.getItem('token');
      await fetch(`/api/users/${userId}/approve`, {
        method: 'PATCH',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ status })
      });
      loadUsers();
    } catch (error) {
      console.error('Error updating user:', error);
    }
  };

  const handleUpdateWallet = async () => {
    if (!walletAmount || !selectedUser) return;

    try {
      const token = localStorage.getItem('token');
      await fetch(`/api/users/${selectedUser.id}/wallet`, {
        method: 'PATCH',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          amount: parseFloat(walletAmount),
          operation: walletOperation
        })
      });
      setShowWalletModal(false);
      setWalletAmount('');
      setSelectedUser(null);
      loadUsers();
    } catch (error) {
      console.error('Error updating wallet:', error);
    }
  };

  if (loading) {
    return <div className="text-center py-8">در حال بارگذاری...</div>;
  }

  return (
    <Card className="p-6">
      <div className="mb-4">
        <h2 className="text-2xl font-bold">مدیریت کاربران</h2>
      </div>

      <Table>
        <TableHeader>
          <TableRow>
            <TableHead>نام</TableHead>
            <TableHead>ایمیل</TableHead>
            <TableHead>شرکت</TableHead>
            <TableHead>کیف پول (گرم)</TableHead>
            <TableHead>وضعیت</TableHead>
            <TableHead>نقش</TableHead>
            <TableHead>عملیات</TableHead>
          </TableRow>
        </TableHeader>
        <TableBody>
          {users.map((user) => (
            <TableRow key={user.id}>
              <TableCell className="font-medium">{user.name}</TableCell>
              <TableCell>{user.email}</TableCell>
              <TableCell>{user.company}</TableCell>
              <TableCell>{user.walletGrams?.toFixed(2)}</TableCell>
              <TableCell>
                {user.status === 'approved' && <Badge className="bg-green-500">تایید شده</Badge>}
                {user.status === 'pending' && <Badge variant="secondary">در انتظار</Badge>}
                {user.status === 'rejected' && <Badge variant="destructive">رد شده</Badge>}
              </TableCell>
              <TableCell>
                <Badge variant={user.role === 'admin' ? 'default' : 'outline'}>
                  {user.role === 'admin' ? 'ادمین' : 'کاربر'}
                </Badge>
              </TableCell>
              <TableCell>
                <div className="flex gap-2">
                  {user.status === 'pending' && (
                    <>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => handleApproveUser(user.id, 'approved')}
                      >
                        <CheckCircle className="h-4 w-4 mr-1" />
                        تایید
                      </Button>
                      <Button
                        size="sm"
                        variant="destructive"
                        onClick={() => handleApproveUser(user.id, 'rejected')}
                      >
                        <XCircle className="h-4 w-4 mr-1" />
                        رد
                      </Button>
                    </>
                  )}
                  {user.role !== 'admin' && (
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => {
                        setSelectedUser(user);
                        setShowWalletModal(true);
                      }}
                    >
                      <Wallet className="h-4 w-4 mr-1" />
                      کیف پول
                    </Button>
                  )}
                </div>
              </TableCell>
            </TableRow>
          ))}
        </TableBody>
      </Table>

      {/* Wallet Management Modal */}
      <Dialog open={showWalletModal} onOpenChange={setShowWalletModal}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>مدیریت کیف پول - {selectedUser?.name}</DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div>
              <Label>موجودی فعلی</Label>
              <div className="text-2xl font-bold">{selectedUser?.walletGrams?.toFixed(2)} گرم</div>
            </div>
            
            <div>
              <Label>نوع عملیات</Label>
              <Select value={walletOperation} onValueChange={setWalletOperation}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="add">افزودن به موجودی</SelectItem>
                  <SelectItem value="set">تنظیم موجودی</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <Label>مقدار (گرم)</Label>
              <Input
                type="number"
                value={walletAmount}
                onChange={(e) => setWalletAmount(e.target.value)}
                placeholder="مثال: 100"
                min="0"
                step="0.01"
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowWalletModal(false)}>
              انصراف
            </Button>
            <Button onClick={handleUpdateWallet}>
              ذخیره
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </Card>
  );
}